/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved.
*/

#ifndef UUID_e4d6e914_4c3f_4066_9ccb_4753481eb4ff
#define UUID_e4d6e914_4c3f_4066_9ccb_4753481eb4ff

/**
 * \file vncsbpserialize.h
 *
 * Declares the VNC CDB and SBP SDK APIs related to SBP payload serialization
 * and deserialization.
 *
 * \see ::VNCSBPSerialize, ::VNCSBPDeserialize
 */

#include <stddef.h>

#include "vnccall.h"
#include "vncint.h"
#include "vnccdbtypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Unique identifier of an object or a field in a structure.
 *
 * To hash a string into a UID, use VNCSBPHash().
 */
typedef vnc_uint32_t VNCSBPUID;

/**
 * \brief Datatypes that may be serialized in an SBP payload.
 */
typedef enum
{
  /**
   * Null type. This value is not defined by the specification, but is
   * provided as a convenience (for variable initialization etc).
   */
  VNCSBPTypeNone           = 0x00,
  /** End-of-structure marker. */
  VNCSBPTypeEnd            = 0x81,
  /** Boolean. Either 0 or 1. */
  VNCSBPTypeBoolean        = 0x82,
  /** 8-bit signed integer. */
  VNCSBPTypeByte           = 0x83,
  /** 16-bit signed integer. */
  VNCSBPTypeShort          = 0x84,
  /** 32-bit signed integer. */
  VNCSBPTypeInt            = 0x85,
  /** 64-bit signed integer. */
  VNCSBPTypeLong           = 0x86,
  /** Single-precision IEEE 754-1985 floating point number. */
  VNCSBPTypeFloat          = 0x87,
  /** Double-precision IEEE 754-1985 floating point number. */
  VNCSBPTypeDouble         = 0x88,
  /** Array of bytes. */
  VNCSBPTypeBytes          = 0x90,
  /**
   * Character string. These are serialized as UTF-16 but are passed via the
   * API as UTF-8.
   */
  VNCSBPTypeString         = 0x91,
  /** Array of primitive values. */
  VNCSBPTypeArray          = 0xa0,
  /** Structure containing fields identified by their own UIDs. */
  VNCSBPTypeStructure      = 0xa1,
  /** Array of structures. */
  VNCSBPTypeStructureArray = 0xa2,
  /** End-of-command marker. */
  VNCSBPTypeEndCommand = 0xb0,
} VNCSBPType;

/**
 * \brief Indicates that a string being passed via the API is NUL-terminated.
 */
#define VNCSBPNullTerminated ((size_t) (-1))

/**
 * \brief Implements the hash function that generates SBP UIDs for strings.
 *
 * \param string The string to be hashed. This parameter is treated as an array
 * of unsigned bytes, rather than as a text string (i.e. its encoding is not
 * relevant to the function of this API).
 * \param stringLength The length of the string in bytes. May be
 * ::VNCSBPNullTerminated, in which case the string is assumed to be
 * NUL-terminated.
 *
 * \return The hashed string.
 *
 * \see VNCSBPUID
 */
typedef vnc_int32_t VNCCALL
VNCSBPHash(const char *string, size_t stringLength);

/**
 * \brief Object that serializes a single SBP payload.
 *
 * To create a VNCSBPSerialize instance, call VNCSBPSerializeCreate().
 * ::VNCSBPSerialize instances are not re-usable; a new ::VNCSBPSerialize 
 * must be created for each payload to be serialized.
 *
 * Use \ref sbp_serialize "the data serialization APIs", such as
 * VNCSBPSerializeInt(), to serialize data into a ::VNCSBPSerialize object. 
 * When the serialization is complete, call VNCSBPSerializeFinalize() to 
 * obtain the address of the serialized data.
 *
 * When the ::VNCSBPSerialize instance is no longer required, call
 * VNCSBPSerializeDestroy() to destroy it.
 *
 * \see VNCSBPSerializeCreate(), VNCSBPSerializeDestroy(),
 * VNCSBPSerializeFinalize(), sbp_serialize
 */
typedef struct VNCSBPSerialize VNCSBPSerialize;

/**
 * \brief Creates and returns a new ::VNCSBPSerialize instance.
 *
 * When the object is no longer required, it must be destroyed with
 * VNCSBPSerializeDestroy().
 *
 * \param ppSerialize On successful return, *ppSerialize is a pointer to the
 * new ::VNCSBPSerialize instance.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorOutOfMemory Not enough memory to complete the 
 * operation.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeCreate(VNCSBPSerialize **ppSerialize);

/**
 * \brief Destroys a ::VNCSBPSerialize instance.
 *
 * Destroying a ::VNCSBPSerialize instance renders invalid the pointer that was
 * returned by VNCSBPSerializeFinalize().
 *
 * \param pSerialize The ::VNCSBPSerialize that should be destroyed.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 *
 * \see VNCSBPSerializeCreate()
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeDestroy(VNCSBPSerialize *pSerialize);

/**
 * \brief Finalizes the serialized payload stored in a ::VNCSBPSerialize
 * instance.
 *
 * After creating a ::VNCSBPSerialize and using the
 * \ref sbp_serialize "data serialization APIs" to add data to it, the
 * application must call VNCSBPSerializeFinalize() to gain access to the
 * serialized data.
 *
 * The memory pointed to by the returned pointer is owned by the
 * ::VNCSBPSerialize instance. Do not alter or free it. Also, do not access the
 * memory after the ::VNCSBPSerialize instance has been destroyed.
 *
 * \param pSerialize The ::VNCSBPSerialize that should be finalized.
 * \param ppData On successful return, *ppData points to the finalized SBP
 * payload buffer.
 * \param pDataSize On successful return, *pDataSize is the size of the
 * finalized SBP payload buffer.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize, pData or pDataSize is NULL.
 * \retval ::VNCCDBErrorSBPSerializeIncomplete The data written to the
 * ::VNCSBPSerialize instance do not form a complete SBP payload. For example, a
 * call to ::VNCSBPSerializeBeginArray() was made, but there was no matching call
 * to VNCSBPSerializeEnd().
 *
 * \see VNCSBPSerializeCreate(), VNCSBPSerializeDestroy()
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeFinalize(VNCSBPSerialize *pSerialize,
                        const vnc_uint8_t **pData,
                        size_t *pDataSize);

/**
 * \defgroup sbp_serialize Data Serialization
 *
 * Use the data serialization APIs to add data into a ::VNCSBPSerialize
 * instance.
 *
 * Each of the data serialization APIs (except for VNCSBPSerializeEnd() has a
 * UID parameter). This should usually be the unique ID of the data being
 * serialized.
 *
 * However, members of ARRAYs and STRUCTURE_ARRAYs do not have their own UIDs.
 * For this reason, when calling a data serialization API to add a member to an
 * ARRAY or STRUCTURE_ARRAY, the application must supply a UID equal to 0. If
 * the application attempts to add a member to an ARRAY or STRUCTURE_ARRAY and
 * supplies a UID that is not 0, then the operation will always fail, and the
 * API will return the error ::VNCCDBErrorSBPSerializeUnexpectedUID.
 *
 * @{
 */

/**
 * \brief Appends a Boolean to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param boolean The Boolean value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A Boolean cannot be added
 * here, because the containing type is either a STRUCTURE_ARRAY or an ARRAY of
 * a type other than Boolean.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the Boolean is a member of an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeBoolean(VNCSBPSerialize *pSerialize,
                       VNCSBPUID uid,
                       vnc_int32_t boolean);

/**
 * \brief Appends a signed byte to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The byte value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A byte cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY of a
 * type other than signed byte.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeByte(VNCSBPSerialize *pSerialize, VNCSBPUID uid, vnc_int8_t n);

/**
 * \brief Appends a signed 16-bit integer to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The short value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A short cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY of a
 * type other than short.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeShort(VNCSBPSerialize *pSerialize, VNCSBPUID uid, vnc_int16_t n);

/**
 * \brief Appends a signed 32-bit integer to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The integer value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType An integer cannot be added
 * here, because the containing type is either a STRUCTURE_ARRAY or an ARRAY of
 * a type other than integer.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeInt(VNCSBPSerialize *pSerialize, VNCSBPUID uid, vnc_int32_t n);

/**
 * \brief Appends a signed 64-bit integer to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The long value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A long cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY of a
 * type other than long.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeLong(VNCSBPSerialize *pSerialize, VNCSBPUID uid, vnc_int64_t n);

/**
 * \brief Appends a single-precision IEEE 754-1985 floating point number to a
 * serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The float value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A float cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY of a
 * type other than float.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeFloat(VNCSBPSerialize *pSerialize, VNCSBPUID uid, vnc_float_t n);

/**
 * \brief Appends a double-precision IEEE 754-1985 floating point number to a
 * serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param n The double value.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A double cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY of a
 * type other than double.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeDouble(VNCSBPSerialize *pSerialize,
                      VNCSBPUID uid,
                      vnc_double_t n);

/**
 * \brief Appends an array of bytes to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param bytes The byte array. Must be non-NULL if byteCount is non-zero.
 * \param byteCount The number of bytes in the array.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL, or byteCount is
 * non-zero and bytes is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A 'bytes' cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeBytes(VNCSBPSerialize *pSerialize,
                     VNCSBPUID uid,
                     const vnc_int8_t *bytes,
                     size_t byteCount);

/**
 * \brief Appends a text string to a serialized SBP payload.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the field, or 0 if the field is a member of an
 * ARRAY or a STRUCTURE_ARRAY.
 * \param string The string, encoded as UTF-8.
 * \param stringLength The length of the string in bytes. May be
 * ::VNCSBPNullTerminated, in which case the string is assumed to be
 * NUL-terminated.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL, or string is NULL
 * and stringLength is non-zero.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A string cannot be added here,
 * because the containing type is either a STRUCTURE_ARRAY or an ARRAY.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is either an ARRAY or a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeString(VNCSBPSerialize *pSerialize,
                      VNCSBPUID uid,
                      const char *string,
                      size_t stringLength);

/**
 * \brief Begins serialization of a STRUCTURE.
 *
 * Each call to VNCSBPSerializeBeginStructure() must be matched by a
 * corresponding call to VNCSBPSerializeEnd(). To add data as members of the
 * structure, call other methods between these two calls.
 *
 * The SDK counts the number of members that are added to the structure and
 * automatically adds the count to the serialized buffer.
 *
 * For example:
 *
   \code

   VNCCDBSDK sdk; // initialized elsewhere
   VNCSBPSerialize *pSerialize; // initialized elsewhere
   const VNCSBPUID uid_s = (*sdk.vncSBPHash)("s", VNCSBPNullTerminated);
   const VNCSBPUID uid_a = (*sdk.vncSBPHash)("a", VNCSBPNullTerminated);
   const VNCSBPUID uid_b = (*sdk.vncSBPHash)("b", VNCSBPNullTerminated);

   // Serialize a structure with two fields { a = 1, b = 2 }
   (*sdk.vncSBPSerializeBeginStructure)(pSerialize, uid_s);
   (*sdk.vncSBPSerializeInt)(pSerialize, uid_a, 1);
   (*sdk.vncSBPSerializeInt)(pSerialize, uid_b, 2);
   (*sdk.vncSBPSerializeEnd)(pSerialize);

   \endcode
 *
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the structure, or 0 if the field is a member of a
 * STRUCTURE_ARRAY.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A structure cannot be added here,
 * because the containing type is an ARRAY.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeBeginStructure(VNCSBPSerialize *pSerialize, VNCSBPUID uid);

/**
 * \brief Begins an SBP packet.
 *
 * This call will start an SBP packet containing one or more SBP 
 * objects. This may then be used as the payload for a SET request,
 * or a GET or SUBSCRIBE response.
 *
 * A call to VNCSBPSerializeBeginPacket() must be matched by a
 * corresponding call to VNCSBPSerializeEnd(). To add objects to the packet, call
 * other methods between these two calls.
 *
 * The SDK counts the number of members that are added to the structure and
 * automatically adds the count to the serialized buffer.
 *
 * For example:
 *
   \code

   VNCCDBSDK sdk; // initialized elsewhere
   VNCSBPSerialize *pSerialize; // initialized elsewhere
   const VNCSBPUID uid_a = (*sdk.vncSBPHash)("a", VNCSBPNullTerminated);
   const VNCSBPUID uid_b = (*sdk.vncSBPHash)("b", VNCSBPNullTerminated);

   // Serialize a packet containing two objects 'a = 1' and 'b = 2'
   (*sdk.vncSBPSerializeBeginPacket)(pSerialize);
   (*sdk.vncSBPSerializeInt)(pSerialize, uid_a, 1);
   (*sdk.vncSBPSerializeInt)(pSerialize, uid_b, 2);
   (*sdk.vncSBPSerializeEnd)(pSerialize);

   \endcode
 *
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the structure, or 0 if the field is a member of a
 * STRUCTURE_ARRAY.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A packet cannot be added
 * within another structure or array.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeBeginPacket(VNCSBPSerialize *pSerialize);


/**
 * \brief Begins serialization of an ARRAY, STRUCTURE_ARRAY, or SBP packet.
 *
 * This single API handles serializes both ARRAYs and STRUCTURE_ARRAYs. If the
 * valid of the type parameter is VNCSBPTypeStructure, then a STRUCTURE_ARRAY
 * is generated; otherwise, an ARRAY is generated.
 *
 * Each call to VNCSBPSerializeBeginArray() must be matched by a corresponding
 * call to VNCSBPSerializeEnd(). To add data as members of the array, call
 * other methods between these two calls.
 *
 * The SDK counts the number of members that are added to the array and
 * automatically adds the count to the serialized buffer.
 *
 * \note The serialized format of ARRAYs and STRUCTURE_ARRAYs does not allow
 * for each member to have its own UID. For this reason, the application must
 * specify a UID of 0 for each member that is added to an ARRAY or
 * STRUCTURE_ARRAY.
 *
 * For example:
 *
   \code

   VNCCDBSDK sdk; // initialized elsewhere
   VNCSBPSerialize *pSerialize; // initialized elsewhere
   const VNCSBPUID uid_array = (*sdk.vncSBPHash)("array", VNCSBPNullTerminated);

   // Serialize an array of 4 ints with values { 1, 2, 3, 4 }
   (*sdk.vncSBPSerializeBeginArray)(pSerialize, uid_array, VNCSBPTypeInt);
   (*sdk.vncSBPSerializeInt)(pSerialize, 0, 1);
   (*sdk.vncSBPSerializeInt)(pSerialize, 0, 2);
   (*sdk.vncSBPSerializeInt)(pSerialize, 0, 3);
   (*sdk.vncSBPSerializeInt)(pSerialize, 0, 4);
   (*sdk.vncSBPSerializeEnd)(pSerialize);

   \endcode
 *
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 * \param uid The UID of the structure, or 0 if the field is a member of a
 * STRUCTURE_ARRAY.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedType A structure cannot be added here,
 * because the containing type is an ARRAY.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedUID A non-zero UID was given, but
 * the containing type is a STRUCTURE_ARRAY.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeBeginArray(VNCSBPSerialize *pSerialize,
                          VNCSBPUID uid,
                          VNCSBPType type);

/**
 * \brief Ends serialization of an ARRAY, a STRUCTURE or a STRUCTURE_ARRAY.
 *
 * This API flags an in-progress ARRAY, STRUCTURE or STRUCTURE_ARRAY as
 * complete. The SDK fills in the count of the members that have been added by
 * the application and, in the case of STRUCTUREs and STRUCTURE_ARRAYs, appends
 * an END marker.
 *
 * \param pSerialize The ::VNCSBPSerialize instance to which to append.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pSerialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory Memory allocation failed.
 * \retval ::VNCCDBErrorSBPSerializeComplete No more data can be written, because
 * the data that has already been written to the ::VNCSBPSerialize instance 
 * forms a complete SBP payload.
 * \retval ::VNCCDBErrorSBPSerializeUnexpectedEnd No ARRAY, STRUCTURE or
 * STRUCTURE_ARRAY is in progress.
 */
typedef VNCCDBError VNCCALL
VNCSBPSerializeEnd(VNCSBPSerialize *pSerialize);

/** @} */

/**
 * \defgroup sbp_deserialize_callbacks Data Deserialization
 *
 * When calling VNCSBPDeserializeCreate(), the application supplies a
 * ::VNCSBPDeserializeCallbacks structure containing pointers to callbacks that
 * the SDK will invoke if it encounters data of the appropriate type.
 *
 * It is legal and safe for the application not to provide a full set of
 * deserialization callbacks. The application should set the unused callback
 * pointers in the ::VNCSBPDeserializeCallbacks structure to NULL. In this case,
 * the SDK will not attempt to invoke the unimplemented callbacks, even if data
 * of the associated type is present. Instead, ::VNCSBPDeserializeExecute()
 * will fail with the error ::VNCCDBErrorSBPDeserializeCallbackNotImplemented.
 *
 * For example, if a particular ::VNCSBPDeserialize instance will only ever
 * deserialize payloads which are ARRAYs of INTs, then the only necessary
 * callbacks are ::VNCSBPDeserializeBeginArrayCallback,
 * ::VNCSBPDeserializeIntCallback and ::VNCSBPDeserializeEndCallback. If a call
 * to VNCSBPDeserializeExecute() encounters an ARRAY of SHORTs instead, then
 * ::VNCCDBErrorSBPDeserializeCallbackNotImplemented is returned.
 *
 * To handle ARRAY or STRUCTURE_ARRAY data, the application must implement both
 * a VNCSBPDeserializeBeginArrayCallback() and a
 * VNCSBPDeserializeEndCallback(). To handle STRUCTURE data, the application
 * must implement both a VNCSBPDeserializeBeginStructureCallback() and a
 * VNCSBPDeserializeEndCallback().
 *
 * VNCSBPDeserializeBooleanCallback(), VNCSBPDeserializeByteCallback(),
 * VNCSBPDeserializeShortCallback(), VNCSBPDeserializeIntCallback(),
 * VNCSBPDeserializeLongCallback(), VNCSBPDeserializeFloatCallback(),
 * VNCSBPDeserializeDoubleCallback(), VNCSBPDeserializeBytesCallback(),
 * VNCSBPDeserializeStringCallback(),
 * VNCSBPDeserializeBeginStructureCallback(),
 * VNCSBPDeserializeBeginArrayCallback(), VNCSBPDeserializeEndCallback()
 *
 * @{
 */

/**
 * \brief Object that deserializes SBP payloads.
 *
 * To create a ::VNCSBPDeserialize instance, call VNCSBPDeserializeCreate(). This
 * method takes as its parameters a table of pointers to
 * \ref sbp_deserialize_callbacks "callbacks" that the SDK will invoke while
 * deserializing each SBP payload.
 *
 * To deserialize an SBP payload, call VNCSBPDeserializeExecute(). The same
 * ::VNCSBPDeserialize instance may be used to deserialize multiple SBP payloads
 * (even from multiple different threads, at the same time). Typically, an
 * application will use one ::VNCSBPDeserialize instance for each type of message
 * to be deserialized, and provide each instance with its own callbacks.
 *
 * When the ::VNCSBPSerialize instance is no longer required, call
 * VNCSBPSerializeDestroy() to destroy it.
 *
 * \see VNCSBPDeserializeCreate(), VNCSBPDeserializeDestroy(),
 * VNCSBPDeserializeExecute()
 */
typedef struct VNCSBPDeserialize VNCSBPDeserialize;

/**
 * \brief Handles BOOLEAN data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param boolean The Boolean value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeBooleanCallback(VNCSBPDeserialize *pDeserialize,
                                 void *pInstanceContext,
                                 void *pExecutionContext,
                                 VNCSBPUID uid,
                                 vnc_int32_t boolean);

/**
 * \brief Handles BYTE data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The byte value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeByteCallback(VNCSBPDeserialize *pDeserialize,
                              void *pInstanceContext,
                              void *pExecutionContext,
                              VNCSBPUID uid,
                              vnc_int8_t n);

/**
 * \brief Handles SHORT data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The short value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeShortCallback(VNCSBPDeserialize *pDeserialize,
                               void *pInstanceContext,
                               void *pExecutionContext,
                               VNCSBPUID uid,
                               vnc_int16_t n);

/**
 * \brief Handles INT data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The int value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeIntCallback(VNCSBPDeserialize *pDeserialize,
                             void *pInstanceContext,
                             void *pExecutionContext,
                             VNCSBPUID uid,
                             vnc_int32_t n);

/**
 * \brief Handles LONG data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The long value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeLongCallback(VNCSBPDeserialize *pDeserialize,
                              void *pInstanceContext,
                              void *pExecutionContext,
                              VNCSBPUID uid,
                              vnc_int64_t n);

/**
 * \brief Handles FLOAT data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The float value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeFloatCallback(VNCSBPDeserialize *pDeserialize,
                               void *pInstanceContext,
                               void *pExecutionContext,
                               VNCSBPUID uid,
                               vnc_float_t n);

/**
 * \brief Handles DOUBLE data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param n The double value.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeDoubleCallback(VNCSBPDeserialize *pDeserialize,
                                void *pInstanceContext,
                                void *pExecutionContext,
                                VNCSBPUID uid,
                                vnc_double_t n);

/**
 * \brief Handles BYTES data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param bytes The byte array. This memory is owned by the SDK; do not alter
 * or free it.
 * \param byteCount The length of the byte array.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeBytesCallback(VNCSBPDeserialize *pDeserialize,
                               void *pInstanceContext,
                               void *pExecutionContext,
                               VNCSBPUID uid,
                               const vnc_int8_t *bytes,
                               size_t byteCount);

/**
 * \brief Handles STRING data.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param string The extracted string data, as a NUL-terminated UTF-8 encoded
 * string.
 * \param stringLength The length of the string in bytes. This length does not
 * include the NUL-terminator that the SDK has added.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeStringCallback(VNCSBPDeserialize *pDeserialize,
                                void *pInstanceContext,
                                void *pExecutionContext,
                                VNCSBPUID uid,
                                const char *string,
                                size_t stringLength);

/**
 * \brief Handles the beginning of a STRUCTURE.
 *
 * This callback is called at the start of each STRUCTURE that is encountered.
 * The end of the STRUCTURE is notified with the
 * VNCSBPDeserializeEndCallback(). Any callbacks between a
 * VNCSBPDeserializeBeginStructureCallback() and its corresponding
 * VNCSBPDeserializeEndCallback() indicate members of that STRUCTURE.
 * Therefore, if the application implements a
 * VNCSBPDeserializeBeginStructureCallback(), then it should always also
 * implement a VNCSBPDeserializeEndCallback().
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param memberCount The number of members of the structure.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeBeginStructureCallback(VNCSBPDeserialize *pDeserialize,
                                        void *pInstanceContext,
                                        void *pExecutionContext,
                                        VNCSBPUID uid,
                                        size_t memberCount);

/**
 * \brief Handles the beginning of an ARRAY or a STRUCTURE_ARRAY.
 *
 * This callback is called at the start of each ARRAY or STRUCTURE_ARRAY that
 * is encountered. The end of the ARRAY or STRUCTURE_ARRAY is notified with the
 * VNCSBPDeserializeEndCallback(). Any callbacks between a
 * VNCSBPDeserializeBeginArrayCallback() and its corresponding
 * VNCSBPDeserializeEndCallback() indicate members of that ARRAY or
 * STRUCTURE_ARRAY. Therefore, if the application implements a
 * VNCSBPDeserializeBeginArrayCallback(), then it should always also implement
 * a VNCSBPDeserializeEndCallback().
 *
 * Because ARRAYs and STRUCTURE_ARRAYs do not allow for their members to have
 * individual UIDs, the UIDs passed to any callbacks for members of the array
 * will all be 0.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. If the data is contained in an ARRAY, then
 * the UID is always 0, because ARRAY members do not have their own UID.
 * \param memberType The type of members of the array. For an ARRAY, this will
 * always be one of ::VNCSBPTypeBoolean, ::VNCSBPTypeShort, ::VNCSBPTypeInt,
 * ::VNCSBPTypeLong, ::VNCSBPTypeFloat, or ::VNCSBPTypeDouble. For a
 * STRUCTURE_ARRAY, this will always be ::VNCSBPTypeStructure.
 * \param memberCount The number of members of the array.
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return V::NCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeBeginArrayCallback(VNCSBPDeserialize *pDeserialize,
                                    void *pInstanceContext,
                                    void *pExecutionContext,
                                    VNCSBPUID uid,
                                    VNCSBPType memberType,
                                    size_t memberCount);

/**
 * \brief Handles the end of a STRUCTURE, an ARRAY or a STRUCTURE_ARRAY.
 *
 * This callback is called at the end of each STRUCTURE, ARRAY or
 * STRUCTURE_ARRAY that is encountered. It indicates that parsing of the
 * innermost structure or array has now finished, and that any further
 * callbacks refer to members of the outer scope.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance that raised the event.
 * \param pInstanceContext The instance context pointer passed to
 * VNCSBPDeserializeCreate().
 * \param pExecutionContext The execution context pointer passed to
 * VNCSBPDeserializeExecute().
 * \param uid The UID of the data. This is the same as the UID passed to the
 * corresponding VNCSBPDeserializeBeginArrayCallback() or
 * VNCSBPDeserializeBeginStructureCallback().
 *
 * \return The SDK does not inspect the returned value. However, for future
 * compatibility, the callback should return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeEndCallback(VNCSBPDeserialize *pDeserialize,
                             void *pInstanceContext,
                             void *pExecutionContext,
                             VNCSBPUID uid);

/**
 * \brief Table of \ref sbp_deserialize_callbacks
 * to be invoked by each call to VNCSBPDeserializeExecute() for a particular
 * ::VNCSBPDeserialize instance.
 *
 * \see VNCSBPDeserializeCreate()
 */
typedef struct
{
    /** Handles BOOLEAN data. */
    VNCSBPDeserializeBooleanCallback *pBooleanCallback;
    /** Handles BYTE data. */
    VNCSBPDeserializeByteCallback *pByteCallback;
    /** Handles SHORT data. */
    VNCSBPDeserializeShortCallback *pShortCallback;
    /** Handles INT data. */
    VNCSBPDeserializeIntCallback *pIntCallback;
    /** Handles LONG data. */
    VNCSBPDeserializeLongCallback *pLongCallback;
    /** Handles FLOAT data. */
    VNCSBPDeserializeFloatCallback *pFloatCallback;
    /** Handles DOUBLE data. */
    VNCSBPDeserializeDoubleCallback *pDoubleCallback;
    /** Handles BYTES data. */
    VNCSBPDeserializeBytesCallback *pBytesCallback;
    /** Handles STRING data. */
    VNCSBPDeserializeStringCallback *pStringCallback;
    /** Handles the beginning of a STRUCTURE. */
    VNCSBPDeserializeBeginStructureCallback *pBeginStructureCallback;
    /** Handles the beginning of an ARRAY or a STRUCTURE_ARRAY. */
    VNCSBPDeserializeBeginArrayCallback *pBeginArrayCallback;
    /** Handles the end of a STRUCTURE, an ARRAY or a STRUCTURE_ARRAY. */
    VNCSBPDeserializeEndCallback *pEndCallback;
} VNCSBPDeserializeCallbacks;

/**
 * \brief Creates a new ::VNCSBPDeserialize instance.
 *
 * When the ::VNCSBPDeserialize instance is no longer required, call
 * VNCSBPDeserializeDestroy() to destroy it.
 *
 * \param ppDeserialize On successful return, *ppDeserialize points to the new
 * ::VNCSBPDeserialize instance.
 * \param pInstanceContext Context pointer permanently associated with this
 * instance of ::VNCSBPDeserialize. This pointer is passed to each of the
 * callbacks associated with this instance. The SDK does not alter the memory
 * pointed to, and does not free it when the VNCSBPDeserialize instance is
 * destroyed.
 * \param pCallbacks Pointer to a ::VNCSBPDeserializeCallbacks structure
 * containing the addresses of callbacks to be invoked by each call to
 * VNCSBPDeserializeExecute(). The memory pointed to is copied by the SDK; it
 * is safe to deallocate *pCallbacks after VNCSBPDeserializeCreate() returns.
 * \param callbacksSize The size of the ::VNCSBPDeserializeCallbacks structure
 * to which pCallbacks points.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter ppDeserialize is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A memory allocation failed.
 *
 * \see ::VNCSBPDeserialize, ::VNCSBPDeserializeCallbacks
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeCreate(VNCSBPDeserialize **ppDeserialize,
                        void *pInstanceContext,
                        const VNCSBPDeserializeCallbacks *pCallbacks,
                        size_t callbacksSize);

/**
 * \brief Destroys a ::VNCSBPDeserialize instance.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance to be destroyed.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pDeserialize is NULL.
 *
 * \see ::VNCSBPDeserialize
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeDestroy(VNCSBPDeserialize *pDeserialize);

/** \brief Returns the instance context from a ::VNCSBPDeserialize instance.
 *
 * \param pDeserialize The ::VNCSBPDeserialize from which to recover the context.
 *
 * \return The instance context pointer supplied to VNCSBPDeserializeCreate().
 *
 * \see ::VNCSBPDeserialize, VNCSBPDeserializeCreate()
 */
typedef void * VNCCALL
VNCSBPDeserializeGetContext(VNCSBPDeserialize *pDeserialize);

/**
 * \brief Deserializes an SBP payload buffer.
 *
 * During the deserialization, the SDK invokes the callbacks that were provided
 * in the call to VNCSBPDeserializeCreate() for this ::VNCSBPDeserialize
 * instance. For example, if the input data is
 *
 * <pre>
   STRUCTURE str
   {
       INT a;
       INT b;
   };
   STRUCTURE_ARRAY<str> s_array = { { 1, 2 }, { 3, 4 } };
   </pre>
 *
 * then the deserializing s_array would generate this sequence of callbacks:
 *
 * \code
   VNCSBPDeserializeBeginArrayCallback(
       uid = hash("s_array"),
       memberType = VNCSBPTypeStructure,
       memberCount = 2);
   VNCSBPDeserializeBeginStructureCallback(
       uid = 0, // zero because array members do not have their own UIDs
       memberCount = 2);
   VNCSBPDeserializeIntCallback(
       uid = hash("a"),
       n = 1);
   VNCSBPDeserializeIntCallback(
       uid = hash("b"),
       n = 2);
   VNCSBPDeserializeEndCallback(
       uid = 0);
   VNCSBPDeserializeBeginStructureCallback(
       uid = 0,
       memberCount = 2);
   VNCSBPDeserializeIntCallback(
       uid = hash("a"),
       n = 3);
   VNCSBPDeserializeIntCallback(
       uid = hash("b"),
       n = 4);
   VNCSBPDeserializeEndCallback(
       uid = 0);
   VNCSBPDeserializeEndCallback(
       uid = hash("s_array"));
   \endcode
 *
 * where hash(s) is the value returned by calling
 *
 * \code
   VNCSBPHash(s, VNCSBPNullTerminated)
   \endcode
 *
 * See \ref sbp_deserialize_callbacks
 * for details of the individual callbacks.
 *
 * \note This function is used for raw SBP structures - to deserialize the
 *       entire payload of an SBP packet, as passed to the SBP callback
 *       functions, use VNCSBPDeserializeExecutePacket() instead.
 *
 * \param pDeserialize The ::VNCSBPDeserialize instance to be used.
 * \param pExecutionContext Context pointer associated with this call to
 * VNCSBPDeserializeExecute(). During this call to VNCSBPDeserializeExecute(),
 * this context pointer and the instance context pointer that was passed to
 * VNCSBPDeserializeCreate() are both passed to each of the callbacks
 * associated with this instance. The SDK does not alter the memory pointed to,
 * and does not free it when the ::VNCSBPDeserialize instance is destroyed.
 * \param serialized The serialized SBP payload data.
 * \param serializedSize The size of the serialized SBP payload data.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pDeserialize is NULL, or serializedSize
 * is non-zero and serialized is NULL.
 * \retval ::VNCCDBErrorFailed The call failed for some unexpected reason.
 * \retval ::VNCCDBErrorSBPDeserializeCallbackNotImplemented The data in the
 * serialized buffer requires a callback that was not provided by the
 * application.
 * \retval ::VNCCDBErrorSBPDeserializeInputTruncated The data in the serialized
 * buffer was used up before the deserialization was completed.
 * \retval ::VNCCDBErrorSBPDeserializeInputTooLong The input data does contain a
 * valid serialized SBP payload, but it also contains extra data at the end.
 * The application may choose not to treat this condition as an error.
 * \retval ::VNCCDBErrorSBPDeserializeInputMalformed The input data is not a
 * valid serialized SBP payload.
 *
 * \see ::VNCSBPDeserialize, ::VNCSBPDeserializeCallbacks
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeExecute(VNCSBPDeserialize *pDeserialize,
                         void *pExecutionContext,
                         const vnc_uint8_t *serialized,
                         size_t serializedSize);


/**
 * \brief Deserializes an SBP packet payload.
 *
 * This method is intended as the top-level call from an SBP callback function
 * to deserialize the payload of a Put message to the SBP Source, or a Get or
 * Subscribe response to the SBP Sink. The data pointer and size from the
 * callback should be passed directly into this function, and the appropriate
 * deserialization callbacks will be invoked for each field of the encoded
 * object.
 *
 * See VNCSBPDeserializeExecute() for more information on how deserialization
 * operates.
 *
 * \note This differs from VNCSBPDeserializeExecute() in that it expects a bare
 *       number of elements followed by the element data, as it occurs in the
 *       SBP packet payload, whereas VNCSBPDeserializeExecute() expects a valid
 *       encoded SBP structure.
 *
 * \param[in] pDeserialize The ::VNCSBPDeserialize instance to be used.
 * \param[in] pExecutionContext Context pointer associated with this call to
 * VNCSBPDeserializeExecutePacket(). During this call, this context pointer and
 * the instance context pointer that was passed to VNCSBPDeserializeCreate()
 * are both passed to each of the callbacks associated with this instance.
 * The SDK does not alter the memory pointed to, and does not free it when the
 * ::VNCSBPDeserialize instance is destroyed.
 * \param serialized The serialized SBP payload data.
 * \param serializedSize The size of the serialized SBP payload data.
 *
 * \retval ::VNCCDBErrorNone Success.
 * \retval ::VNCCDBErrorInvalidParameter pDeserialize is NULL, or serializedSize
 * is non-zero and serialized is NULL.
 * \retval ::VNCCDBErrorFailed The call failed for some unexpected reason.
 * \retval ::VNCCDBErrorSBPDeserializeCallbackNotImplemented The data in the
 * serialized buffer requires a callback that was not provided by the
 * application.
 * \retval ::VNCCDBErrorSBPDeserializeInputTruncated The data in the serialized
 * buffer was used up before the deserialization was completed.
 * \retval ::VNCCDBErrorSBPDeserializeInputTooLong The input data does contain a
 * valid serialized SBP payload, but it also contains extra data at the end.
 * The application may choose not to treat this condition as an error.
 * \retval ::VNCCDBErrorSBPDeserializeInputMalformed The input data is not a
 * valid serialized SBP payload.
 *
 * \see ::VNCSBPDeserialize, ::VNCSBPDeserializeCallbacks
 */
typedef VNCCDBError VNCCALL
VNCSBPDeserializeExecutePacket(VNCSBPDeserialize *pDeserialize,
                               void *pExecutionContext,
                               const vnc_uint8_t *serialized,
                               size_t serializedSize);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_e4d6e914_4c3f_4066_9ccb_4753481eb4ff) */
